--------------------------------------------------------------------------
-- sd_flash_if.vhd
--
-- Author  : Nial Stewart, Nial Stewart Developments Ltd
--          www.nialstewartdevelopments.co.uk
-- Date   : 09/05/05
--
--------------------------------------------------------------------------
-- Simple interface to initialise the SD card into SPI mode, this verifies
-- that all the lines are connected & working.
--
--------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


entity sd_card_if is
port (
      rst             : in std_logic;
      clk             : in std_logic;  -- 80MHz system clock

      -- Control Signals
      init_start      : in std_logic;
      init_fin        : out std_logic;

      init_track      : out std_logic_vector(7 downto 0);
      -- SD Card signals
      sd_cs           : out std_logic;
      sd_dout         : out std_logic;
      sd_din          : in std_logic;
      sd_clk          : out std_logic;

      debug           : out std_logic_vector(2 downto 0)
      );
end sd_card_if;

architecture rtl of sd_card_if is

signal txrx_go        : std_logic;

-- Initialisation process

signal init_cs        : std_logic;
signal init_data      : std_logic_vector(7 downto 0);
signal transfer_count : integer range 0 to 255;
signal init_txrx_go   : std_logic;
type init_state_type is (POR, PO_DELAY, CMND0, RESPONSE_0, CMND1, RESPONSE_1, FINAL_CLKS);
signal init_state     : init_state_type;
signal init_sub_state : std_logic_vector(1 downto 0);
signal init_track_int : unsigned(7 downto 0);


-- Data Transfer process signals
type transfer_state_type is (IDLE, TRANSFERRING);
signal transfer_state : transfer_state_type;
signal sd_clk_int     : std_logic;
signal tx_data        : std_logic_vector(7 downto 0);
signal rx_data        : std_logic_vector(7 downto 0);
signal shift_reg_out  : std_logic_vector(7 downto 0);
signal shift_reg_in   : std_logic_vector(7 downto 0);
signal clk_tick       : std_logic;
signal clk_tick_count : unsigned(3 downto 0);
signal bit_count      : integer range 0 to 7;
signal txrx_fin       : std_logic;
signal txrx_fin_a1    : std_logic;


begin


-------------------------------------------------------------------------------
-- Initialisation process
--
-- Init procedure is ...
-- CS High
-- Send 10 * 0xFF to give card time to power up
-- CS LOW
-- Send 0x40, 0, 0, 0, 0, 0x95
-- Look for 0x01 back - this indicates the card's in Idle state, the
-- Loop doing..
--    Send 0x41, 0, 0, 0, 0, 0xFF
--    Look for 0x00
-- ..a load of times. When you get 0x00 back it shows the card's
-- out of idle in SPI mode.
-- CS High.
-------------------------------------------------------------------------------


process(clk,rst)
begin
if(rst = '1') then
  init_cs         <= '1';
  init_state      <= POR;
  init_data       <= (others => '1');
  transfer_count  <= 9;
  init_track_int  <= (others => '0');
  init_fin       <= '0';
elsif(rising_edge(clk)) then
  case init_state is
    when POR =>
      transfer_count  <= 9;
      init_cs         <= '1';
      init_data       <= (others => '1');
      if(init_start = '1') then
        init_state      <= PO_DELAY;
        init_txrx_go    <= '1';
        init_fin        <= '0';
      end if;

    when PO_DELAY =>
      if(txrx_fin = '1') then
        if(transfer_count = 1) then
        --  init_cs <= '0';
        end if;

        if(transfer_count = 0) then
          init_state    <= CMND0;
          init_data     <= x"40";
          init_txrx_go  <= '1';
          init_cs <= '0';

          transfer_count <= 6;
        else
          init_txrx_go  <= '1';
          transfer_count <= transfer_count - 1;
        end if;
      else
        init_txrx_go <= '0';
      end if;

    when CMND0 =>
      init_cs       <= '0';
      if(txrx_fin = '1') then
        if(transfer_count = 1) then
          init_data  <= (others => '1'); -- Not really needed, but easier in the testbench
          init_state <= RESPONSE_0;
          init_txrx_go <= '1';
        elsif(transfer_count = 2) then
          init_data <= x"95";
          init_txrx_go <= '1';
          transfer_count <= 1;
        else
          init_data <= x"00";
          init_txrx_go <= '1';
          transfer_count <= transfer_count - 1;
        end if;
      else
        init_txrx_go <= '0';
      end if;

    when RESPONSE_0 =>
      if(txrx_fin = '1') then
        if(shift_reg_in = x"01") then
          init_state <= CMND1; -- XXXX
          init_data  <= x"41";
          init_txrx_go <= '1';
          transfer_count <= 6;
        else
          init_txrx_go <= '1';
        end if;
      else
        init_txrx_go <= '0';
      end if;

    when CMND1 =>
      if(txrx_fin = '1') then
        if(transfer_count = 1) then
          init_state <= RESPONSE_1;
          transfer_count <= 31;
          init_txrx_go <= '1';
          init_data  <= (others => '1');
        elsif(transfer_count = 2) then
          init_data <= x"FF";
          init_txrx_go <= '1';
          transfer_count <= 1;
        else
          init_data <= x"00";
          init_txrx_go <= '1';
          transfer_count <= transfer_count - 1;
        end if;
      else
        init_txrx_go <= '0';
      end if;

    when RESPONSE_1 =>
      if(txrx_fin = '1') then
        init_txrx_go  <= '1';

        if(shift_reg_in = x"00") then
          init_state    <= FINAL_CLKS;
          init_cs       <= '1';
        else
          if(transfer_count = 0) then
            init_state <= CMND1;
            init_data  <= x"41";
            init_txrx_go <= '1';
            transfer_count <= 6;
         else
            transfer_count <= transfer_count - 1;
         end if;
        end if;
      else
        init_txrx_go <= '0';
      end if;

    when OTHERS => -- FINAL_CLKS

      init_txrx_go <= '0';
      if(txrx_fin = '1') then
        init_state <= POR;
        init_fin   <= '1';
        init_track_int <= init_track_int + 1;
      end if;

  end case;
end if;
end process;

init_track <= std_logic_vector(init_track_int);

-------------------------------------------------------------------------------
-- Low level data transfer process to shift data in and out.
-------------------------------------------------------------------------------


txrx_go <= init_txrx_go;
tx_data <= init_data;


process(clk,rst)
begin
if(rst = '1') then
  transfer_state <= IDLE;
  shift_reg_out <= (others => '1'); -- This may be important for initialisation
  shift_reg_in  <= (others => '0');
  sd_clk_int    <= '1';
  txrx_fin      <= '0';
  clk_tick      <= '0';
  clk_tick_count<= (others => '0');
elsif(rising_edge(clk)) then
  case transfer_state is
    when IDLE =>
      txrx_fin    <= '0';
      if(txrx_go = '1') then
        transfer_state <= TRANSFERRING;
        shift_reg_out <= tx_data;
        bit_count     <= 7;
        sd_clk_int    <= '0';
      else
        sd_clk_int        <= '1';
      end if;

    when OTHERS => -- TRANSFERRING
      if(clk_tick   = '1') then
        if(sd_clk_int = '1') then
          if(bit_count = 0) then
            if(txrx_go = '1') then
              shift_reg_out <= tx_data;
              bit_count     <= 7;
              sd_clk_int    <= not(sd_clk_int);
            else
              transfer_state <= IDLE;
            end if;
          else
            sd_clk_int <= not(sd_clk_int);
            shift_reg_out(7 downto 1) <= shift_reg_out(6 downto 0);
            shift_reg_out(0) <= '1';
            bit_count <= bit_count - 1;
          end if;
        else
          sd_clk_int <= not(sd_clk_int);
          shift_reg_in (0) <= sd_din;
          shift_reg_in(7 downto 1) <= shift_reg_in(6 downto 0);
        end if;
      end if;

      if(clk_tick_count = x"1" and sd_clk_int = '1' and bit_count = 0) then
        txrx_fin <= '1';
      else
        txrx_fin <= '0';
      end if;
  end case;

  if(transfer_state = IDLE) then
    clk_tick_count <= x"E";
    clk_tick       <= '0';
  elsif(clk_tick_count = x"0") then
    clk_tick_count <= x"F";
    clk_tick <= '1';
  else
    clk_tick_count <= clk_tick_count - 1;
    clk_tick <= '0';
  end if;
end if;
end process;

sd_dout <= shift_reg_out(7); -- Everything is sent MSB first.
sd_clk  <= sd_clk_int;

sd_cs <= init_cs;






end rtl;
