-------------------------------------------------------------------------------
-- rs485_if.vhd
--
-- Author : Nial Stewart, Nial Stewart Developments Ltd.
--          www.nialstewartdevelopments.co.uk
-- Date   : 01/03/08
--
-------------------------------------------------------------------------------
-- This is a simple module to drive the RS485 interfaces on the NSD GPIB
-- for hardware test debug.
--
-- A direction bit in controls which channel is Tx'r and which is Rx'r,
-- an enable bit allows the data to be continuously sent when it's active high.
-- The test sends a byte, waits 'till a byte is received and checks it's
-- the same as that transmitted and repeats this, stopping if the received
-- byte is incorrect.
-- The data is sent asynchronously, 1 start bit 8 data and 1 stop, as with
-- RS232 comms.
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity rs485_test is
generic(
      CLK_TICKS_PER_BIT   : in integer range 1 to 512 := 80 -- Default for 80MHz system clock = 1 Mbps
      );
port (
      clk                 : in  std_logic;
      rst                 : in  std_logic;

      -- Config bits
      direction           : in std_logic; -- '0' = 0 Tx & 1 Rx, '1' = 1 Tx & 0 Rx.
      enable              : in std_logic; -- From the Uart interface, controls when the
                                          -- test runs

      fail                : out std_logic;

      -- RS 485 Interfaces
      rs485_0_rx          : in std_logic;
      rs485_0_tx          : out std_logic;
      rs485_0_rx_en_n     : out std_logic;
      rs485_0_tx_en       : out std_logic;

      rs485_1_rx          : in std_logic;
      rs485_1_tx          : out std_logic;
      rs485_1_rx_en_n     : out std_logic;
      rs485_1_tx_en       : out std_logic

     );
end rs485_test;

architecture rtl of rs485_test is


-- Data tx and rx check process
type top_state_type is (IDLE,SEND_DATA,GET_DATA,COMPARE);
signal top_state        : top_state_type;

signal dout             : unsigned(7 downto 0);
signal sent             : std_logic_vector(7 downto 0);
signal din              : std_logic_vector(7 downto 0);

-- RX Process Signals
type rx_state_type is (IDLE, START_CHECK, RXING, END_DELAY);
signal rx_state         : rx_state_type;

signal rx_bit_count     : integer range 1 to 10;
signal rx_shift_reg     : std_logic_vector(9 downto 0);
signal rxd_byte         : std_logic_vector(7 downto 0);
signal rx_clk_count     : integer range 1 to 1024;
signal got_byte         : std_logic; -- flag back to master process that a byte's been received.

signal rx_data          : std_logic;
signal rx_data_d        : std_logic;
signal d_in             : std_logic;

-- Tx Process Signals
type tx_state_type is (IDLE, SENDING, END_WAIT);
signal tx_state         : tx_state_type;
signal tx_shift_reg     : std_logic_vector(9 downto 0);
signal tx_bit_count     : integer range 1 to 10;
signal tx_clk_count     : integer range 1 to 1024;
signal send_byte        : std_logic;
signal next_op_byte     : std_logic_vector(7 downto 0);

begin

-- Direction = '0' = ch0 tx
rs485_0_tx_en   <= not(direction);
rs485_0_rx_en_n <= not(direction);

rs485_1_tx_en   <= direction;
rs485_1_rx_en_n <= direction;


-------------------------------------------------------------------------------
-- Top level process.
-- This process sends a byte, waits until it's received and checks that what's
-- received matches what was sent. Tx data is a simple increment, change to
-- PRBS pattern?
-------------------------------------------------------------------------------

process(clk,rst)
begin
if(rst = '1') then
  dout        <= (others => '0');
  sent        <= (others => '0');
  din         <= (others => '0');
  fail        <= '0';
  top_state   <= IDLE;

elsif(rising_edge(clk)) then
  case(top_state) is
    when IDLE =>
      if(enable = '1') then
        if(tx_state = IDLE) then      -- Wait until the Tx process is ready
          top_state   <= SEND_DATA;
        end if;
      else
        fail  <= '0'; -- Reset pass/fail flag when test stopped.
      end if;

    when SEND_DATA =>         -- Send byte.
      dout          <= dout  + 1;
      sent          <= std_logic_vector(dout);
      next_op_byte  <= std_logic_vector(dout);
      send_byte     <= '1';
      top_state     <= GET_DATA;

    when GET_DATA =>          -- Wait for byte to be rx'd
      send_byte     <= '0';
      if(got_byte = '1') then
        top_state <= COMPARE;
      end if;

    when OTHERS =>
      if(rxd_byte /= sent) then
        fail <= '1';
      end if;
      if(enable = '1') then
        if(tx_state = IDLE) then
          top_state <= SEND_DATA;
        else
          top_state <= IDLE;
        end if;
      else
        top_state <= IDLE;
      end if;
  end case;
end if;
end process;



-------------------------------------------------------------------------------
-- The output process. Simply waits for the send_byte = '1' instruction then
-- reads the input and starts transmitting.
-------------------------------------------------------------------------------

process(clk,rst)
begin
if(rst = '1') then
  tx_state         <= IDLE;
  tx_shift_reg     <= (others => '1');
  tx_bit_count     <= 1;
  tx_clk_count     <= 1;
elsif(rising_edge(clk)) then

  case tx_state is
    when IDLE =>
      if(send_byte = '1') then
        tx_shift_reg(9)         <= '0'; -- Start bit
        for i in 0 to 7 loop            -- Swap bit order, lsb goes first.
          tx_shift_reg(8 - i) <= next_op_byte(i);
        end loop;
        tx_shift_reg(0)          <= '1'; -- This is the stop bit.
        tx_bit_count            <= 1;
        tx_clk_count            <= CLK_TICKS_PER_BIT;    -- Set delay to 1 bit period.
        tx_state                <= SENDING;
      else
        tx_shift_reg <= (others => '1');
      end if;

    when SENDING =>
      if(tx_clk_count = 1) then
        tx_shift_reg(9 downto 1) <= tx_shift_reg(8 downto 0);
        tx_shift_reg(0) <= '1';

        if(tx_bit_count = 10) then                -- Clock out 9 bit periods.
          tx_state <= END_WAIT;
          tx_clk_count <= CLK_TICKS_PER_BIT;
          tx_bit_count <= 1;
        else
          tx_bit_count <= tx_bit_count + 1;
          tx_clk_count <= CLK_TICKS_PER_BIT;
        end if;
      else
        tx_clk_count <= tx_clk_count - 1;
      end if;

    when OTHERS => --END_WAIT -- Wait for an extra 1 bit periods between characters
      if(tx_clk_count = 1) then
        if(tx_bit_count = 1) then
          tx_state <= IDLE;
        else
          tx_bit_count <= tx_bit_count + 1;
          tx_clk_count <= CLK_TICKS_PER_BIT;
        end if;
      else
        tx_clk_count <= tx_clk_count - 1;
      end if;

  end case;

  if(direction = '0') then
    rs485_0_tx <= tx_shift_reg(9);
  else
    rs485_1_tx <= tx_shift_reg(9);
  end if;

end if;
end process;




-------------------------------------------------------------------------------
-- Rx Process. This process just sits waiting for characters and flags to the
-- master process when one's received properly.
-------------------------------------------------------------------------------

rx_data <= rs485_1_rx when direction = '0' else rs485_0_rx;

process(clk,rst)
begin
if(rst = '1') then
  rx_data_d     <= '1';
  d_in          <= '1';
  rx_state      <= IDLE;
  rx_bit_count  <= 1;
  rx_shift_reg  <= (others => '0');
  rx_clk_count  <= CLK_TICKS_PER_BIT;
  rxd_byte      <= (others => '0');
  got_byte      <= '0';
elsif(rising_edge(clk)) then
  rx_data_d <= rx_data;
  d_in      <= rx_data_d;

  if(enable = '0') then -- Don't want to start receiving anything 'till the test starts!
    rx_state <= IDLE;
  else

    case rx_state is
      when IDLE =>
        got_byte <= '0';
        if(d_in = '0') then                       -- If start bit detected...
          rx_clk_count <= CLK_TICKS_PER_BIT/2;    -- ..wait half a bit period...
          rx_state <= START_CHECK;
        end if;

      when START_CHECK =>
        if(rx_clk_count = 1) then
          if(d_in = '0') then                     -- ..if it's still '0' then wait for..
            rx_clk_count <= CLK_TICKS_PER_BIT;    -- ...one bit period and start receiving bits
            rx_bit_count <= 1;
            rx_state     <= RXING;
          else
            rx_state <= IDLE;
          end if;
        else
          rx_clk_count <= rx_clk_count - 1;
        end if;

      when RXING =>
        if(rx_clk_count = 1) then
          rx_shift_reg(0) <= d_in;
          rx_shift_reg(9 downto 1) <= rx_shift_reg(8 downto 0);

          if(rx_bit_count = 9) then               -- Wait 'till mid stop bit
            rx_clk_count <= CLK_TICKS_PER_BIT;    -- At end wait for one more bit period
            rx_state <= END_DELAY;                -- so the rest of the stop bit isn't seen as a start bit
          else
            rx_clk_count <= CLK_TICKS_PER_BIT;
            rx_bit_count <= rx_bit_count + 1;
          end if;
        else
          rx_clk_count <= rx_clk_count - 1;
        end if;

      when OTHERS => -- END_DELAY
          for j in 0 to 7 loop                    -- Swap bit order, lsb is tx'd first.
            rxd_byte(j) <= rx_shift_reg(8 - j);
          end loop;
          got_byte <= '1';
          rx_state <= IDLE;

    end case;
  end if;

end if;
end process;
















end rtl;



