--------------------------------------------------------------------------
--
-- gpib_fpga_2.vhd
--
-- FPGA design to drive the different modules in the Rev2 General Purpose
-- Interface Board
--
--
-- Autor: Nial Stewart, Nial Stewart Developments Ltd
-- Date : Aug/08
--
--------------------------------------------------------------------------


library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

use work.pkg_gpib.all;

entity gpib_fpga_2 is
port (
      clk_50MHz           : in std_logic;  -- 50MHz Starter board crystal


      -- Interfaces to the GPIB

      -- ADC Interface
      adc1_din            : out std_logic;
      adc1_sclk           : out std_logic;
      adc1_cs_n           : out std_logic;
      adc1_dout           : in std_logic;

      -- DAC Interace
      dac1_sclk           : out std_logic;
      dac1_din            : out std_logic;
      dac1_synch_n        : out std_logic;
      dac1_ldac_n         : out std_logic;

      -- SD Card Interface
      sd_cmd              : inout std_logic;  -- Data to card
      sd_clk              : out std_logic;    -- Clock
      sd_data0            : in std_logic;     -- Data from card
      sd_data3            : inout std_logic;  -- Card Select

      -- Vinculum Intface
      --vinc_ad             : inout std_logic_vector(7 downto 0);
      --vinc_ac             : inout std_logic_vector(5 downto 0);
      --vinc_rs_n           : out std_logic;
      --vinc_prog_n         : out std_logic;

      slave_usb_data      : inout std_logic_vector(7 downto 0);
      usb_rxf_n           : in std_logic;
      usb_txe_n           : in std_logic;
      usb_pwren_n         : in std_logic;
      usb_wr              : out std_logic;
      usb_rd              : out std_logic;

      -- 1 Wire Interfaces
      one_wire_data       : inout std_logic_vector(3 downto 1);

      -- LEDs
      leds                : out std_logic_vector(7 downto 0);

      -- RS232 Interfaces
      rs232_0_tx          : out std_logic;
      rs232_0_rx          : in std_logic;
      rs232_1_tx          : out std_logic;
      rs232_1_rx          : in std_logic;
      rs232_2_tx          : out std_logic;
      rs232_2_rx          : in std_logic;

      -- RS485 Interfaces
      rs485_0_rx          : in std_logic;
      rs485_0_tx          : out std_logic;
      rs485_0_rx_en_n     : out std_logic;
      rs485_0_tx_en       : out std_logic;

      rs485_1_rx          : in std_logic;
      rs485_1_tx          : out std_logic;
      rs485_1_rx_en_n     : out std_logic;
      rs485_1_tx_en       : out std_logic;

      -- SPI Interfaces
      spi_0_cs1_n         : out std_logic;
      spi_0_cs2_n         : out std_logic;
      spi_0_sdo           : out std_logic;
      spi_0_sdi           : in std_logic;
      spi_0_clk           : out std_logic;

      spi_1_cs1_n         : out std_logic;
      spi_1_cs2_n         : out std_logic;
      spi_1_sdo           : out std_logic;
      spi_1_sdi           : in std_logic;
      spi_1_clk           : out std_logic;

      -- General Purpose IO
      gpio1               : inout std_logic_vector(7 downto 0);
      gpio2               : inout std_logic_vector(7 downto 0)

      );
end gpib_fpga_2;




architecture rtl of gpib_fpga_2 is

-- 80MHz clock and reset signals to be used throughout
signal clk              : std_logic;
signal rst              : std_logic := '0';
signal pll_locked       : std_logic := '1';
signal rst_count        : unsigned(5 downto 0) := "000000";

-- Misc signals
signal dac1_busy        : std_logic;
signal dac1_value       : std_logic_vector(15 downto 0);
signal dac1_wr          : std_logic;

signal usb_led1         : std_logic_vector(7 downto 0);
signal usb_led2         : std_logic_vector(7 downto 0);
signal usb_mstr_state   : std_logic_vector(7 downto 0);
signal usb_lwlvl_state  : std_logic_vector(7 downto 0);
-- ADC interface signals
signal adc1_ip_range    : std_logic;
signal adc1_seq_end     : std_logic_vector(2 downto 0);
signal adc_ch0          : std_logic_vector(9 downto 0);
signal adc_ch1          : std_logic_vector(9 downto 0);
signal adc_ch2          : std_logic_vector(9 downto 0);
signal adc_ch3          : std_logic_vector(9 downto 0);
signal adc_ch4          : std_logic_vector(9 downto 0);
signal adc_ch5          : std_logic_vector(9 downto 0);
signal adc_ch6          : std_logic_vector(9 downto 0);
signal adc_ch7          : std_logic_vector(9 downto 0);
signal adc_update       : std_logic;

-- Type definition for the One wire interface module.
--type temp_array is array (1 to 10) of std_logic_vector(7 downto 0);
signal temperature      : temp_array;

-- SD Card interface signals
signal sd_chip_sel      : std_logic;
signal sd_data_in       : std_logic;

signal sd_init          : std_logic;
signal sd_init_fin      : std_logic;

-- RS485 Interface test signals
signal rs485_direction  : std_logic;
signal rs485_enable     : std_logic;
signal rs485_fail       : std_logic;

-- GPIO Interfaces
signal gpio1_dir        : std_logic_vector(7 downto 0);
signal gpio1_dout       : std_logic_vector(7 downto 0);
signal gpio1_din        : std_logic_vector(7 downto 0);

signal gpio2_dir        : std_logic_vector(7 downto 0);
signal gpio2_dout       : std_logic_vector(7 downto 0);
signal gpio2_din        : std_logic_vector(7 downto 0);

-- LED signals
signal led_control      : std_logic_vector(1 downto 0);
signal led_dout         : std_logic_vector(7 downto 0);

-- Control Uart Local Bus signals...
signal lb_address       : std_logic_vector(6 downto 0);
signal lb_d_out         : std_logic_vector(15 downto 0);
signal lb_d_in          : std_logic_vector(15 downto 0);
signal lb_wr            : std_logic;
signal lb_rd            : std_logic;
signal byte_addr        : std_logic_vector(7 downto 0);


-- Debug & test
signal led_count        : unsigned(31 downto 0);

signal last_test_reg    : std_logic_vector(15 downto 0);
signal default_reg      : std_logic_vector(15 downto 0);

signal sd_dout          : std_logic;
signal sd_clk_int       : std_logic;
signal sd_status        : std_logic_vector(7 downto 0);
signal init_track       : std_logic_vector(7 downto 0);


begin

--------------------------------------------------------------------------
-- Instantiate the PLL and generate a chip wide reset , disabled
-- some period afte the PLL has locked.
--------------------------------------------------------------------------

pll_inst:entity work.pll_50in_80out
PORT MAP (
    inclk0   => clk_50mhz,
    c0       => clk,
    locked   => pll_locked
    );


process(pll_locked,clk)
begin
if(pll_locked = '0') then
  rst_count <= "000000";
  rst       <= '1';
elsif(rising_edge(clk)) then
  if(rst = '1') then
    rst_count <= rst_count + 1;
  end if;
  if(rst_count = "101010") then
    rst <= '0';
  end if;
end if;
end process;


-------------------------------------------------------------------------------
-- Simple LED counter process
-------------------------------------------------------------------------------

process(clk,rst)
begin
if(rst = '1') then
 led_count <= (others => '0');
elsif(rising_edge(clk)) then
  led_count <= led_count + 1;
end if;
end process;

-- LED output is either an incrementing count, the contents of the
-- led_dout register (address 0x20) or all flashing on and off. This
-- is set by the led_control register (0x21).

leds <= not(led_dout) when led_control = "10"
   else not(std_logic_vector(led_count(31 downto 24))) when led_control = "00"
   else "11111111" when (led_control = "01" and led_count(26) = '1')
   else "00000000";


-------------------------------------------------------------------------------
-- Slave USB interface
-------------------------------------------------------------------------------

slve_usb:entity work.ft245r_if
port map(
      clk         => clk, -- in std_logic;

      -- interface lines to usb module
      rxf_n       => usb_rxf_n, -- in std_logic; -- '0' = data available to read
      txe_n       => usb_txe_n, -- in std_logic; -- '0' = output fifo isn't full
      data        => slave_usb_data, -- inout std_logic_vector(7 downto 0); -- Data in from USB if IC
      --data        => open, -- inout std_logic_vector(7 downto 0); -- Data in from USB if IC
      wr          => usb_wr, -- out std_logic; -- Write on rising edge out
      rd_n        => usb_rd, -- out std_logic; -- Reads current data and fetches next byte from fifo.

      pwren_n     => usb_pwren_n, -- in std_logic; -- Power enable from the USB interface IC = Reset for Interface

      led1        => usb_led1, -- out std_logic_vector(7 downto 0);
      led2        => usb_led2, -- out std_logic_vector(7 downto 0)
      mstr_state  => usb_mstr_state , -- out std_logic_vector(7 downto 0);
      lwlvl_state => usb_lwlvl_state -- out std_logic_vector(7 downto 0)
      );



-------------------------------------------------------------------------------
-- DAC Interface Instantiation
-------------------------------------------------------------------------------

dac_inst:entity work.ad5318_if
port map(
      clk         => clk, -- in std_logicc
      rst         => rst, -- in std_logic;
      dac_wr_data => dac1_value,  -- in std_logic_vector(15 downto 0);
      dac_wr      => dac1_wr,     -- in std_logic;
      sclk        => dac1_sclk, -- out std_logic;
      sdout       => dac1_din, -- out std_logic;
      synch       => dac1_synch_n, -- out std_logic;
      ldac_n      => dac1_ldac_n, -- out std_logic
      busy        => dac1_busy
      );


-------------------------------------------------------------------------------
-- ADC Interface Instantiation
-------------------------------------------------------------------------------

adc1_ip_range <= '1';
adc1_seq_end  <= "111";

adc_ints:entity work.ad7918_if
port map(
      clk         => clk, -- in  std_logic;
      rst         => rst, -- in  std_logic;
      ip_range    => adc1_ip_range, -- in std_logic; -- '0' = 0->2* Vref (ie 5V), '1' = 0->2.5V
      seq_end     => adc1_seq_end , -- in std_logic_vector(2 downto 0);
      adc_sclk    => adc1_sclk, -- out std_logic;
      adc_cs_n    => adc1_cs_n, -- out std_logic;
      adc_din     => adc1_din, -- out  std_logic;
      adc_dout    => adc1_dout, -- in std_logic;
      ch_0        => adc_ch0   , -- out std_logic_vector(9 downto 0);
      ch_1        => adc_ch1   , -- out std_logic_vector(9 downto 0);
      ch_2        => adc_ch2   , -- out std_logic_vector(9 downto 0);
      ch_3        => adc_ch3   , -- out std_logic_vector(9 downto 0);
      ch_4        => adc_ch4   , -- out std_logic_vector(9 downto 0);
      ch_5        => adc_ch5   , -- out std_logic_vector(9 downto 0);
      ch_6        => adc_ch6   , -- out std_logic_vector(9 downto 0);
      ch_7        => adc_ch7   , -- out std_logic_vector(9 downto 0);
      ops_valid   => adc_update -- out std_logic
     );


-------------------------------------------------------------------------------
-- One wire Interface Module
-- This module drives the interface to three one wire DS1821 temperature
-- sensors. See the module comments for more details.
-------------------------------------------------------------------------------

one_wire_inst:entity work.ds1821_if_master
port map(
      rst           => rst, -- in std_logic;
      clk           => clk, -- in std_logic;  -- 80MHz system clock
      temp          => temperature, -- out temp_array;
      dq            => one_wire_data -- inout std_logic_vector(3 downto 1)
      );


-------------------------------------------------------------------------------
-- SD Card Interface
-- This is a simple module that just initialises the SD card to SPI mode.
-------------------------------------------------------------------------------

sd_if_inst:entity work.sd_card_if
port map(
      rst             => rst, -- in std_logic;
      clk             => clk, -- in std_logic;  -- 80MHz system clock
      init_start      => sd_init, -- in std_logic;
      init_fin        => sd_init_fin, -- out std_logic;
      init_track      => init_track, -- out std_logic_vector(7 downto 0);
      sd_cs           => sd_chip_sel, -- out std_logic;
      sd_dout         => sd_dout, -- out std_logic;
      sd_din          => sd_data_in, -- in std_logic;
      sd_clk          => sd_clk_int, -- out std_logic;
      debug           => open -- out std_logic_vector(2 downto 0)
      );

sd_data3 <= sd_chip_sel;
sd_data_in <= sd_data0;
sd_cmd <= sd_dout;
sd_clk <= sd_clk_int;


-------------------------------------------------------------------------------
-- RS485 Test Interface
-- This simply loops writing a byte, waits until it's received and checks that the
-- Tx'd and Rx'd values match. The direction bit sets the direction, enable
-- enables the test and fail_pass_n  is an indicator that
-- interface lines are physically OK.
-------------------------------------------------------------------------------

rs485_if:entity work.rs485_test
generic map(
      CLK_TICKS_PER_BIT   => 8 -- in integer range 1 to 512 := 80 -- Default for 80MHz system clock = 1 Mbps
      )
port map(
      clk                 => clk, -- in  std_logic;
      rst                 => rst, -- in  std_logic;
      direction           => rs485_direction, -- in std_logic; -- '0' = 0 Tx & 1 Rx, '1' = 1 Tx & 0 Rx.
      enable              => rs485_enable, -- in std_logic; -- From the Uart interface, controls when the
      fail                => rs485_fail, -- out std_logic;
      rs485_0_rx          => rs485_0_rx     , -- in std_logic;
      rs485_0_tx          => rs485_0_tx     , -- out std_logic;
      rs485_0_rx_en_n     => rs485_0_rx_en_n, -- out std_logic;
      rs485_0_tx_en       => rs485_0_tx_en, -- out std_logic;
      rs485_1_rx          => rs485_1_rx     , -- in std_logic;
      rs485_1_tx          => rs485_1_tx     , -- out std_logic;
      rs485_1_rx_en_n     => rs485_1_rx_en_n, -- out std_logic;
      rs485_1_tx_en       => rs485_1_tx_en  -- out std_logic
     );

-------------------------------------------------------------------------------
-- GPIO Interfaces
-------------------------------------------------------------------------------

-- Very simple interface allows individual bit direction control as in
-- most microcontroller ports.

-- Allow individual pin direction control for two ports
opt_dir:for i in 0 to 7 generate
  gpio1(i) <= gpio1_dout(i) when gpio1_dir(i) = '1' else 'Z';
  gpio2(i) <= gpio2_dout(i) when gpio2_dir(i) = '1' else 'Z';
end generate;

-- Inputs just read ports

gpio1_din <= gpio1;
gpio2_din <= gpio2;




-----------------------------------------------------------------
--  Test Uart Instantiation
--
-- The following module interfaces with RS232 port 0, this has a
-- uart front end for transmitting and receiving characters, there
-- is also a stat machine for interpreting the received data to
-- drive the local bus that feeds the registers driving the various
-- interfaces.
-----------------------------------------------------------------

inst_nsd_uart:entity work.nsd_uart
generic map(
      CLK_TICKS_PER_BIT => 694 -- in integer range 1 to 1024 := 573
      )
port map(
      clk               => clk, -- in std_logic
      rst               => rst, -- in std_logic
      rx_data           => rs232_0_rx, -- in std_logic
      tx_data           => rs232_0_tx, -- out std_logic
      lb_address        => lb_address, -- out std_logic_vector(6 downto 0)
      lb_d_out          => lb_d_out, -- out std_logic_vector(15 downto 0)
      lb_d_in           => lb_d_in, -- in std_logic_vector(15 downto 0)
      lb_wr             => lb_wr, -- out std_logic
      lb_rd             => lb_rd -- out std_logic
);

-- This loops Uart1 & 2 together to allow them to be tested easily.
rs232_2_tx <= rs232_1_rx;
rs232_1_tx <= rs232_2_rx;


-------------------------------------------------------------------------------
-- Embedded register read/write process;
-- This section emulates an FPGA internal Read/ Write processes.
-------------------------------------------------------------------------------

-- The top bit of the address word is used for read/write flag.
-- Create an 8 bit address so we can use hex address notation.
byte_addr <= '0' & lb_address(6 downto 0);

-- Write process

process(clk,rst)
begin
if(rst = '1') then
  dac1_value      <= (others => '0');
  dac1_wr         <= '0';

  sd_init         <= '0';

  last_test_reg   <= (others => '0');
  default_reg     <= (others => '0');

  rs485_direction <= '0';
  rs485_enable    <= '0';

  gpio1_dout      <= (others => '0');
  gpio1_dir       <= (others => '0');
  gpio2_dout      <= (others => '0');
  gpio2_dir       <= (others => '0');

  led_dout        <= (others => '0');
  led_control     <= (others => '0');


elsif(rising_edge(clk)) then
  if(lb_wr = '1') then
    case byte_addr is

      when x"00" =>
        dac1_value <= lb_d_out;
        dac1_wr    <= '1';

      when x"0A" =>
        rs485_enable    <= lb_d_out(0);
        rs485_direction <= lb_d_out(1);

      when x"14" =>
        sd_init <= '1';

      when x"16" =>
        gpio1_dout <= lb_d_out(7 downto 0);
      when x"17" =>
        gpio1_dir <= lb_d_out(7 downto 0);

      when x"18" =>
        gpio2_dout <= lb_d_out(7 downto 0);
      when x"19" =>
        gpio2_dir <= lb_d_out(7 downto 0);

      when x"20" =>
        led_dout     <= lb_d_out(7 downto 0);

      when x"21" =>
        led_control <= lb_d_out(1 downto 0);

      when x"7F" =>
        last_test_reg <= lb_d_out;

      when OTHERS =>
        default_reg <= lb_d_out;

    end case;
  else
    dac1_wr  <= '0';
    sd_init  <= '0';
  end if;
end if;
end process;


-- Read Process

process(clk,rst)
begin
if(rst = '1') then
  lb_d_in <= (others => '0');

elsif(rising_edge(clk)) then
  if(lb_rd = '1') then
    case byte_addr is

      when x"01" =>
        lb_d_in <= "000000" & adc_ch0;

      when x"02" =>
        lb_d_in <= "000000" & adc_ch1;

      when x"03" =>
        lb_d_in <= "000000" & adc_ch2;

      when x"04" =>
        lb_d_in <= "000000" & adc_ch3;

      when x"05" =>
        lb_d_in <= "000000" & adc_ch4;

      when x"06" =>
        lb_d_in <= "000000" & adc_ch5;

      when x"07" =>
        lb_d_in <= "000000" & adc_ch6;

      when x"08" =>
        lb_d_in <= "000000" & adc_ch7;

      when x"0A" =>
        lb_d_in <= x"000" & "000" & rs485_fail;

      when x"10" =>
        lb_d_in <= x"00" & temperature(1);

      when x"11" =>
        lb_d_in <= x"00" & temperature(2);

      when x"12" =>
        lb_d_in <= x"00" & temperature(3);

      when x"14" =>
        lb_d_in <= init_track & x"0" & "000" & sd_init_fin;

      when x"16" =>
        lb_d_in <= x"00" & gpio1_din;

      when x"17" =>
        lb_d_in <= x"00" & gpio1_dir;

      when x"18" =>
        lb_d_in <= x"00" & gpio2_din;

      when x"19" =>
        lb_d_in <= x"00" & gpio2_dir;

      when x"7F" =>
        lb_d_in <= last_test_reg;

      when OTHERS =>
        lb_d_in <= default_reg;

    end case;
  end if;
end if;
end process;



end rtl;


