-------------------------------------------------------------------------------
-- ft245r_if.vhd
--
-- Author : Nial Stewart, Nial Stewart Developments Ltd.
--          www.nialstewartdevelopments.co.uk
-- Date   : The ancient depths of time....
--
--
-------------------------------------------------------------------------------
-- A VHDL module to act as the interface between the NSD USB interface board and
-- any embedded registers in a design. Registers can be  written to and read from
-- although not all readable registers can be written to (ie status registers).
--
-- This is a very simple implementation of a USB interface that is not designed to
-- be critically robust, ie on a read no check is made to ensure the FT245BM
-- write FIFO has space.
--
--
--
-- This module consists of 3 main processes as shown here
--
--
--
--                    ~~~~~~~~~~~~~~
--                    ~            ~
--                    ~ USB driver ~---------
--                    ~    IC      ~         |
--                    ~~~~~~~~~~~~~~         |
--                         ^  |              |
--                         |  V              V
--      ------------------------------------------------------------
--        FPGA             ^  | Control Sigs |
--                         |  V + data       |
--                    ~~~~~~~~~~~~~~         |
--                    ~            ~         |
--                    ~ Interface  ~         |
--                    ~  Process   ~         |
--                    ~~~~~~~~~~~~~~         |
--                         ^  |              |
--                         |  V              | Full/Empty
--                    ~~~~~~~~~~~~~~         | flags
--                    ~            ~         |
--                    ~  Protocol  ~<--------
--                    ~  Process   ~
--                    ~~~~~~~~~~~~~~
--                         ^  |
--                         |  V
--                    ~~~~~~~~~~~~~~
--                    ~  Register  ~
--                    ~   Rd_Wr    ~
--                    ~  Process   ~
--                    ~~~~~~~~~~~~~~
--                    ^ ^ ^ ^ | | | |
--                    | | | | V V V V
--                    ...Registers ...
--
--
-- By structuring the design like this the Interface process is the only thing which
-- must be modified when the target device speed changes, the Protocol process is the
-- only thing to be changed if added interface functionality is needed and the Register
-- Rd_Wr process is the only thing to be modified if registers are added or deleted.
--
--
-- NOTE: This Interface process has been written for a target design clocked at 80MHz. It should
-- work with a target design running slower than 80MHz, but data throuput will
-- be slower than the optimum available.
-- If the target speed is reduced (or increased) the Interface Process will
-- have to be modified to meet the FT245BR timings. Please refer to the FTDI FT245R
-- data sheet for the interface timing reqts if you want to modify this interface
-- (www.ftdichip.com)
--
--------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;


entity ft245r_if is
port( clk         : in std_logic;

      -- interface lines to usb module
      rxf_n       : in std_logic; -- '0' = data available to read
      txe_n       : in std_logic; -- '0' = output fifo isn't full
      data        : inout std_logic_vector(7 downto 0); -- Data in from USB if IC
      wr          : out std_logic; -- Write on rising edge out
      rd_n        : out std_logic; -- Reads current data and fetches next byte from fifo.

      pwren_n     : in std_logic; -- Power enable from the USB interface IC

      led1        : out std_logic_vector(7 downto 0);
      led2        : out std_logic_vector(7 downto 0);

      mstr_state  : out std_logic_vector(7 downto 0);
      lwlvl_state : out std_logic_vector(7 downto 0)
      );
end ft245r_if;

architecture rtl of ft245r_if is

signal rst        : std_logic;


-- The following are used to re-time and re-name asynchronous signals txe_n and
-- rxf_n (which are ridiculously named for their functionality).
signal txe_n_d    : std_logic;
signal tx_full    : std_logic;  -- '1' = tx buffer full so can't write more
signal rxf_n_d    : std_logic;
signal rx_empty    : std_logic; -- '1' = rx buffer empty so can't read more

signal reg1_int   : std_logic_vector(7 downto 0);
signal reg2_int   : std_logic_vector(7 downto 0);
signal reg3_int   : std_logic_vector(7 downto 0);
signal reg4_int   : std_logic_vector(7 downto 0);
signal reg5_int   : std_logic_vector(7 downto 0);
signal reg6_int   : std_logic_vector(7 downto 0);
signal reg7_int   : std_logic_vector(7 downto 0);
signal reg8_int   : std_logic_vector(7 downto 0);
signal reg9_int   : std_logic_vector(7 downto 0);
signal reg10_int  : std_logic_vector(7 downto 0);
signal reg11_int  : std_logic_vector(7 downto 0);
signal reg12_int  : std_logic_vector(7 downto 0);
signal reg13_int  : std_logic_vector(7 downto 0);
signal reg14_int  : std_logic_vector(7 downto 0);
signal reg15_int  : std_logic_vector(7 downto 0);
signal reg16_int  : std_logic_vector(7 downto 0);
signal reg17_int  : std_logic_vector(7 downto 0);
signal reg18_int  : std_logic_vector(7 downto 0);
signal reg19_int  : std_logic_vector(7 downto 0);
signal reg20_int  : std_logic_vector(7 downto 0);
signal led1_int   : std_logic_vector(7 downto 0);
signal led2_int   : std_logic_vector(7 downto 0);

signal last_reg   : std_logic_vector(7 downto 0);

signal data_out   : std_logic_vector(7 downto 0);
signal dout_enable: std_logic;

--
--  The following are associated with the Protocol process.
--

type   states_typ is (IDLE,CMND_RD,PRE_ADDR,ADDR_RD,PRE_VAL,VAL_READ,READING,RD_END_WAIT,WRITING,
                      STATE10,STATE11,STATE12,STATE13,STATE14,STATE15,STATE16);
signal prot_state : states_typ;

signal command    : std_logic_vector(7 downto 0);
signal address    : std_logic_vector(7 downto 0);
signal value      : std_logic_vector(7 downto 0);

constant WRITE    : std_logic_vector(7 downto 0) := x"A0";
constant READ     : std_logic_vector(7 downto 0) := x"50";


--
--  The following signals are associated with the Interface process that
--  the FT245BM interface signals
--

type if_state_typ is(IDLE,RD1,RD2,RD3,RD4,WAIT1,WAIT2,FIN,WR1,WR2,WR3,WR4,
                     WAIT3,WAIT4,DUM3,DUM4,DUM5,DUM6); -- Using an even power 2 declared states for safety.
signal if_state   : if_state_typ;

signal start_wr   : std_logic;
signal start_rd   : std_logic;
signal rx_data    : std_logic_vector(7 downto 0);
signal tx_data    : std_logic_vector(7 downto 0);
signal rdwr_fin   : std_logic;
signal wait_count : integer range 0 to 63;




signal  register_rd : std_logic;
signal  register_wr : std_logic;


begin

rst <= pwren_n;

-- Re-time in and rename the asynch fifo status flags to something sensible.
process(clk,rst)
begin
if (rst = '1') then
  txe_n_d <= '0';
  tx_full <= '0';
  rxf_n_d <= '1';
  rx_empty <= '1';
elsif(rising_edge(clk)) then
  txe_n_d <= txe_n;
  tx_full <= txe_n_d;
  rxf_n_d <= rxf_n;
  rx_empty <= rxf_n_d;
end if;
end process;




--------------------------------------------------------------------------------
--
--  Protocol process.
--
--  This process uses the process above to read and write data from the USB
--  interface and the rd_wr process to access the registers.
--  We start by waiting for a character to be received. This is assigned to
--  command, the next character is read and assigned to address. If the
--  command = x"XA" the value is then read and the write performed. If the
--  command = x"X5" the appropriate register value is written to the USB
--  interface.
--
--  Need the 'PRE_XXX' states to check there's data available before
--  proceeding.
--
--  This process could be easily modified to include a burst read or
--  burst write command.
--
--------------------------------------------------------------------------------


protocol:process(clk,rst)
begin
if(rst = '1') then
  start_rd <= '0';
  start_wr <= '0';
  prot_state <= IDLE;

  register_rd <= '0';
  register_wr <= '0';


elsif(rising_edge(clk)) then
  start_rd <= '0';
  start_wr <= '0';

  case prot_state is
    when IDLE =>
      mstr_state <= (others => '0');
      if (rx_empty = '0') then        -- If the receive buffer's not empty get a character
        start_rd <= '1';
        prot_state <= CMND_RD;
      end if;

    when CMND_RD =>
      mstr_state <= x"01";
      if (rdwr_fin = '1') then        -- When the first char's got...
        command <= rx_data;
        prot_state <= PRE_ADDR;       -- ..get the address..
      end if;

    when PRE_ADDR =>                  -- ..wait for data then get address...
      mstr_state <= x"02";
      if (rx_empty = '0') then
        start_rd <= '1';
        prot_state <= ADDR_RD;
      end if;

    when ADDR_RD =>
      mstr_state <= x"03";
      if (rdwr_fin = '1') then        -- When the address is received....
        address <= rx_data;
        if(command = WRITE) then
          prot_state <= PRE_VAL;
        else                          -- ..if it's a write get the data...
          prot_state <= READING;
          register_rd <= '1';         -- ..else do the read of the local bus.
        end if;
      end if;

    when PRE_VAL =>                   -- Wait for the value to be written to appear
      if (rx_empty = '0') then        -- then get it..
        start_rd <= '1';
        prot_state <= VAL_READ;
      end if;
      mstr_state <= x"04";

    when VAL_READ =>                  -- Get the write value then trigger..
      if(rdwr_fin = '1') then
        value <= rx_data;
        prot_state <= WRITING;        -- .. the write.
        register_wr <= '1';           -- Reg write signal active during the write state.
      end if;
      mstr_state <= x"05";

    when READING =>               -- This sets the read data to tx_data..
      start_wr <= '1';            -- ..write the data out over the usb interface
      prot_state <= RD_END_WAIT;
      register_rd <= '0';
      mstr_state <= x"06";

    when RD_END_WAIT =>
      if(rdwr_fin = '1') then     -- Wait until the read value's been written to.
        prot_state <= IDLE;       -- the USB if then back to IDLE.
      end if;
      mstr_state <= x"07";

    when WRITING =>                     -- Single clock process

      register_wr <= '0';               -- De-assert the write signal and ..
      prot_state <= IDLE;               -- ..back to IDLE.
      mstr_state <= x"08";

    when others =>
      prot_state <= IDLE;
      mstr_state <= x"09";

  end case;

end if;
end process;

--------------------------------------------------------------------------------
--
--  USB Device Interface Process.
--
--  This in the process which drives the FT245R control lines to allow
--  read and write accesses.
--
--  From an IDLE state an 8 bit read or write is initiated using the 'start_wr'
--  or 'start_rd' flags from the protocol process. The completion of a read or
--  write is signalled back with rdwr_fin. This process waits long enough to
--  allow the FT245 tx_full or rx_empty flags to go active before rdwr_fin
--  goes active. This allows the protocol process to check it can proceed with
--  its next byte immediately a rd/wr is finished.
--------------------------------------------------------------------------------

interface_proc:process(clk,rst)
begin
if (rst = '1') then
  if_state <= IDLE;
  wr <= '0'; -- Inactive state
  rd_n <= '1'; -- Inactive state
  rdwr_fin <= '0';
  dout_enable <= '0';

elsif(rising_edge(clk)) then

  case if_state is
    when IDLE=>
      if(start_rd = '1') then
        rd_n <= '0';
        if_state <= RD1;
      elsif(start_wr = '1') then
        wr <= '1';
        if_state <= WR1;
        data_out <= tx_data;
        dout_enable <= '1';
      end if;
      lwlvl_state <= x"00";

    when RD1 =>
      if_state <= RD2;
      lwlvl_state <= x"01";
    when RD2 =>
      if_state <= RD3;
      wait_count  <= 31;
      lwlvl_state <= x"02";
    when RD3 =>
      if(wait_count = 0) then
        if_state <= RD4;
        lwlvl_state <= x"03";
      else
        wait_count <= wait_count - 1;
      end if;
    when RD4 =>
      rd_n <= '1';
      if_state <= WAIT1;
      rx_data <= data;
      lwlvl_state <= x"04";

    when WR1 =>
      if_state <= WR2;
      lwlvl_state <= x"05";
    when WR2 =>
      if_state <= WR3;
      lwlvl_state <= x"06";
    when WR3 =>
      if_state <= WR4;
      lwlvl_state <= x"07";
    when WR4 =>
      lwlvl_state <= x"08";
      if_state <= WAIT1;
      wr <= '0';            -- Note the wr cycle terminates in WAIT1
                            -- when the data out is tri-stated. This is
                            -- because the FT245BM needs >10ns hold time from
                            -- wr going low.

    when WAIT1 =>           -- Need these four wait states because
      if_state <= WAIT2;    -- of the max delay in the RXF_n and TXE_n
      dout_enable <= '0';   -- signals being asserted at the end of
      wait_count  <= 63;
      lwlvl_state <= x"09";
    when WAIT2 =>           -- the read or write cycle, and the fact that
      --if_state <= WAIT3;    -- they're re-timed in. Waiting here
      if(wait_count = 0) then
        if_state <= FIN;
        rdwr_fin <= '1';
      else
        wait_count <= wait_count - 1;
      end if;
      lwlvl_state <= x"0A";
    when WAIT3 =>           -- means the protocol process can check
      if_state <= WAIT4;    -- the signals again as soon as rdwr_fin
      lwlvl_state <= x"0B";
    when WAIT4 =>           -- goes active.
      if_state <= FIN;
      rdwr_fin <= '1';
      lwlvl_state <= x"0C";
    when FIN =>
      if_state <= IDLE;
      rdwr_fin <= '0';
      lwlvl_state <= x"0D";
    when others =>
      if_state <= IDLE;
      lwlvl_state <= x"0E";
  end case;
end if;
end process;

data <= data_out when dout_enable = '1' else "ZZZZZZZZ";
led2(0) <= dout_enable;
--------------------------------------------------------------------------------
--
--  Local Bus RD_WR Process.
--
--  This process is used by the top level process to read and write data from and
--  to the local bus registers.
--
--------------------------------------------------------------------------------

rd_wr_process:process(clk,rst)
begin
if(rst = '1') then
  tx_data <= (others => '0');
elsif(rising_edge(clk)) then

  if(register_rd = '1') then
    case address is
      when x"01" =>
        tx_data <= reg1_int;
      when x"02" =>
        tx_data <= reg2_int;
      when x"03" =>
        tx_data <= reg3_int;
      when x"04" =>
        tx_data <= reg4_int;
      when x"05" =>
        tx_data <= reg5_int;
      when x"06" =>
        tx_data <= reg6_int;
      when x"07" =>
        tx_data <= reg7_int;
      when x"08" =>
        tx_data <= reg8_int;
      when x"09" =>
        tx_data <= reg9_int;
      when x"0A" =>
        tx_data <= reg10_int;
      when x"0B" =>
        tx_data <= reg11_int;
      when x"0C" =>
        tx_data <= reg12_int;
      when x"0D" =>
        tx_data <= reg13_int;
      when x"0E" =>
        tx_data <= reg14_int;
      when x"0F" =>
        tx_data <= reg15_int;
      when x"10" =>
        tx_data <= reg16_int;
      when x"11" =>
        tx_data <= reg17_int;
      when x"12" =>
        tx_data <= reg18_int;
      when x"13" =>
        tx_data <= reg19_int;
      when x"14" =>
        tx_data <= reg20_int;
      when x"15" =>
        tx_data <= led1_int;
      when x"16" =>
        tx_data <= led2_int;

      when x"FF" =>
        tx_data <= last_reg;
      when others =>
        tx_data <= x"34";
    end case;
  end if;


  if(register_wr = '1') then
    case address is
      when x"01" =>
        led1_int <= value;
        reg1_int <= value;
      when x"02" =>
        reg2_int <= value;
      when x"03" =>
        reg3_int <= value;
      when x"04" =>
        reg4_int <= value;
      when x"05" =>
        reg5_int <= value;
      when x"06" =>
        reg6_int <= value;
      when x"07" =>
        reg7_int <= value;
      when x"08" =>
        reg8_int <= value;
      when x"09" =>
        reg9_int <= value;
      when x"0A" =>
        reg10_int <= value;
      when x"0B" =>
        reg11_int <= value;
      when x"0C" =>
        reg12_int <= value;
      when x"0D" =>
        reg13_int <= value;
      when x"0E" =>
        reg14_int <= value;
      when x"0F" =>
        reg15_int <= value;
      when x"10" =>
        reg16_int <= value;
      when x"11" =>
        reg17_int <= value;
      when x"12" =>
        reg18_int <= value;
      when x"13" =>
        reg19_int <= value;
      when x"14" =>
        reg20_int <= value;
      when x"15" =>
        led1_int <= value;
      when x"16" =>
        led2_int <= value;
      when x"FF" =>
        last_reg <= value;

      when others =>
        led2_int <= value;
    end case;

  end if;

end if;
end process;



-- Assign internal signals to ports

led1    <= led1_int;
--led2    <= led2_int;


end rtl;



