--------------------------------------------------------------------------
-- ds1821_if.vhd
--
-- Author : Nial Stewart, Nial Stewart Developments Ltd
--          www.nialstewartdevelopments.co.uk
-- Date   : 09/05/2005
--
--
--------------------------------------------------------------------------
-- This is a simple interface to read temepratures from a series of
-- ds1821 digital thermometers/thermostats (the thermostat functionality
-- isn't used).
--
-- A 'master' process generates the control signals that are used by a
-- series of slave modules that blindly act on the master signals then
-- output the result.
--------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

use work.pkg_gpib.all;

entity ds1821_if_master is
port (
      rst           : in std_logic;
      clk           : in std_logic;  -- 80MHz system clock

      temp          : out temp_array;
      -- 1 wire interface only needs 1 wire!
      dq            : inout std_logic_vector(3 downto 1)
      );
end ds1821_if_master;

architecture rtl of ds1821_if_master is


-------------------------------------------------------------------------------
-- High level process signals
-------------------------------------------------------------------------------

type cntrl_state_type   is (RSET, RST_INIT, START_CONVERSION,LONG_WAIT,READ_INIT, READ_COMMAND, GET_RESULT);
signal cntrl_state      : cntrl_state_type;
signal cntrl_sub_state  : std_logic_vector(2 downto 0);

signal long_delay_count : unsigned(25 downto 0);

constant HALF_SECOND    : unsigned(25 downto 0) := "10" & x"6259FF";  -- Values for 80MHz clock
constant LONG_ZERO      : unsigned(25 downto 0) := "00" & x"000000";

signal init             : std_logic;
signal byte_out         : std_logic_vector(7 downto 0);
signal wr_byte          : std_logic;
signal rd_byte          : std_logic;

-------------------------------------------------------------------------------
-- Low level process signals
-------------------------------------------------------------------------------

type line_control_state_typ is (IDLE, INITIALISE, WRITE_BYTE, READ_BYTE);
signal line_control_state : line_control_state_typ;
signal sub_state        : std_logic_vector(3 downto 0);

signal timer              : unsigned(15 downto 0);

-- Timings all calculated for a 50MHz system clock
constant ZERO             : unsigned(15 downto 0) := x"0000";
constant ONE_US           : unsigned(15 downto 0) := x"004F";
constant TWO_US           : unsigned(15 downto 0) := x"009F";
constant FIVE_US          : unsigned(15 downto 0) := x"018F";
constant TWELVE_US        : unsigned(15 downto 0) := x"03BF";
constant SEVENTY_US       : unsigned(15 downto 0) := x"15DF";
constant FIVE_CENT_US     : unsigned(15 downto 0) := x"9C3F";

signal master_dout        : std_logic;
signal din                : std_logic_vector(3 downto 1);
signal shift_reg          : std_logic_vector(7 downto 0);
signal done               : std_logic;
signal bit_count          : integer range 0 to 8;

signal reset              : std_logic;

signal detect_sensor     : std_logic;
signal shift_bit_in       : std_logic;
signal op_reading         : std_logic;


-------------------------------------------------------------------------------
-- Slave modules signals
-------------------------------------------------------------------------------

signal temperature      : std_logic_vector(7 downto 0);

begin
reset <= '0';

dq(3) <= '0' when (master_dout = '0') else 'Z';
din(3) <= dq(3);

dq(2) <= '0' when (master_dout = '0') else 'Z';
din(2) <= dq(2);

dq(1) <= '0' when (master_dout = '0') else 'Z';
din(1) <= dq(1);
-------------------------------------------------------------------------------
-- High level state control
--
-- This controls the lower level process that generates the interface
-- signals. It also generates the 'latch_temp' signals to the slave
-- processes when the temepratures have been registered in.
-------------------------------------------------------------------------------



process(clk,rst)
begin
if(rst = '1') then
  cntrl_state       <= RSET;
  cntrl_sub_state   <= "000";
  long_delay_count  <= (others => '0');
  init              <= '0';
  byte_out          <= (others => '0');
  wr_byte           <= '0';
  rd_byte           <= '0';
  op_reading        <= '0';
elsif(rising_edge(clk)) then
  case cntrl_state is

    when RSET =>
      cntrl_state <= RST_INIT;

    when RST_INIT =>
      op_reading <= '0';
      if(cntrl_sub_state = "000") then
        init <= '1';                    -- Send an initialisation sequence...
        cntrl_sub_state <= "001";
      else
        init <= '0';
        if(done = '1') then             -- ..when that's finished start conversion...
          cntrl_sub_state <= "000";
          cntrl_state <= START_CONVERSION;
        end if;
      end if;

    when START_CONVERSION =>
      if(cntrl_sub_state = "000") then
        byte_out <= x"EE";              -- Write out 0xEE to start a conversion...
        wr_byte <= '1';
        cntrl_sub_state <= "001";
      else
        wr_byte <= '0';
        if(done = '1') then             -- ..when that's done wait for half a second...
          cntrl_sub_state <= "000";
          cntrl_state <= LONG_WAIT;
          long_delay_count <= HALF_SECOND;
        end if;
      end if;

    when LONG_WAIT =>
      if(long_delay_count = LONG_ZERO) then -- ..after half a second
        cntrl_state <= READ_INIT;
        cntrl_sub_state <= "000";
      else
        long_delay_count <= long_delay_count - 1;
      end if;

    when READ_INIT =>
      if(cntrl_sub_state = "000") then
        init <= '1';                      -- Send an init sequence.....
        cntrl_sub_state <= "001";
      else
        init <= '0';
        if(done = '1') then               -- ..when that's done...
          cntrl_sub_state <= "000";
          cntrl_state <= READ_COMMAND;
        end if;
      end if;

    when READ_COMMAND =>
      if(cntrl_sub_state = "000") then
        byte_out <= x"AA";                -- Write 0xAA "Read Temperature"...
        wr_byte <= '1';
        cntrl_sub_state <= "001";
      else
        wr_byte <= '0';
        if(done = '1') then               -- ...then...
          cntrl_state <= GET_RESULT;
          cntrl_sub_state <= "000";
        end if;
      end if;

    when OTHERS =>  --GET_RESULT
      if(cntrl_sub_state = "000") then
        rd_byte <= '1';                   -- Read the result byte in...
        cntrl_sub_state <= "001";
      else
        rd_byte <= '0';
        if(done = '1') then               -- ..when finished...
          op_reading  <= '1';             -- ..trigger the slave modules to output their sensor
          cntrl_state <= RST_INIT;        -- ..present and temperature results out then back to the start.
          cntrl_sub_state <= "000";
          long_delay_count <= HALF_SECOND;
        end if;
      end if;

  end case;
end if;
end process;




-------------------------------------------------------------------------------
-- Low level interface process
--
-- This process generates the master_dout signal and indicates to the
-- slave modules when to latch data in.
-------------------------------------------------------------------------------


process(clk,rst)
begin
if(rst = '1') then
  line_control_state <= IDLE;
  timer              <= (others => '0');
  master_dout        <= '1';
  sub_state          <= "0000";
  done               <= '0';
  shift_reg          <= (others => '0');
  detect_sensor     <= '0';
  shift_bit_in       <= '0';

elsif(rising_edge(clk)) then
  if(reset = '1') then
    master_dout <= '1';
    line_control_state <= IDLE;
    sub_state          <= "0000";
    done        <= '0';
    shift_reg   <= (others => '0');
  else
    case line_control_state is

      when IDLE =>
        master_dout <= '1';
        done        <= '0';
        sub_state <= "0000";
        if(init = '1') then                 -- If an init is triggered then load the
          timer <= FIVE_CENT_US;            -- counter and initialise
          line_control_state <= INITIALISE;
          master_dout              <= '0';
        elsif(wr_byte = '1') then           -- Same for Write byte..
          timer <= TWO_US;                  -- (start with 2 US pulse low)
          master_dout              <= '0';
          shift_reg <= byte_out;
          line_control_state <= WRITE_BYTE;
          bit_count          <= 0;
        elsif(rd_byte = '1') then           -- ..and read byte
          timer <= TWO_US;                  -- (start with 2 US pulse low)
          master_dout              <= '0';
          line_control_state <= READ_BYTE;
          bit_count          <= 0;

        end if;

      when INITIALISE =>
        if(sub_state = "0000") then       -- Initialisation is 500uS low output on dout
          if(timer = ZERO) then
            master_dout <= '1';           -- .. then release...
            timer <= SEVENTY_US;          -- Wait for 70 uS..
            sub_state <= "0001";
          else
            timer <= timer - 1;
          end if;
        elsif(sub_state = "0001") then
          if(timer = ZERO) then           -- .. then look for the presence of a sensor (it'll
            detect_sensor <= '1';        -- have pulled the data line low if it's there).
            timer <= FIVE_CENT_US;        -- Wait for another 500 uS..
            sub_state <= "0010";
          else
            timer <= timer - 1;
          end if;
        else
          detect_sensor <= '0';
          if(timer = ZERO) then
            done <= '1';                  -- ..signal we've finished and..
            line_control_state <= IDLE;   -- .. back to IDLE
          else
            timer <= timer - 1;
          end if;
        end if;

      when WRITE_BYTE =>                  -- WRITE BYTE
        if(sub_state = "0000") then       -- D_out has been set low and count set to 2uS.
          if(timer = ZERO) then           -- When 2uS up ..
            if(shift_reg(0) = '1') then   -- .. if next bit to be sent is high..
              master_dout <= '1';         -- .. set op high (else it's left low)..
            end if;
            timer <= SEVENTY_US;          --.. for another 70uS.
            sub_state <= "0001";
          else
            timer <= timer - 1;
          end if;

        elsif(sub_state = "0001") then
          if(timer = ZERO) then
            master_dout <= '1';           -- Set op high to finish bit and..
            timer <= FIVE_US;             -- .. wait 5uS between bits
            sub_state <= "0010";
          else
            timer <= timer - 1;
          end if;

        else
          if(timer = ZERO) then
            if(bit_count = 7) then        -- If that's the end of the byte..
              done <= '1';                -- ..signal it's done and..
              line_control_state <= IDLE; -- ..go back to IDLE..
            else
              timer <= TWO_US;            -- ..else start 2uS low start for next bit and..
              shift_reg(6 downto 0) <= shift_reg(7 downto 1); -- ..shift it down ..
              bit_count <= bit_count + 1;

              sub_state <= "0000";
              master_dout <= '0';
            end if;
          else
            timer <= timer - 1;
          end if;
        end if;

      when OTHERS =>                      -- READ_BYTE
        if(sub_state = "0000") then       -- D_out has been set low and count set to 2uS.
          if(timer = ZERO) then           -- When 2uS up ..
            sub_state <= "0001";
            master_dout <= '1';
            timer <= TWELVE_US;           -- ..wait 12uS..
          else
            timer <= timer - 1;
          end if;

        elsif(sub_state = "0001") then
          if(timer = ZERO) then           -- ..then..
            sub_state <= "0010";
            shift_bit_in <= '1';          -- ..register in the next bit..
            timer <= SEVENTY_US;          -- ..and wait 70uS, well past the end of the bit
          else
            timer <= timer - 1;
          end if;

        else
          shift_bit_in <= '0';
          if(timer = ZERO) then           -- At end of 70uS wait..
            if(bit_count = 7) then        -- ..if we've reached the end of the byte..
              done  <= '1';               -- ..signal we've finished..
              line_control_state <= IDLE; -- ..and return to IDLE.
            else
              timer <= TWO_US;            -- ..else start 2uS low at start of next read bit
              sub_state <= "0000";
              master_dout <= '0';
              bit_count <= bit_count + 1;
            end if;
          else
            timer <= timer - 1;
          end if;

        end if;

    end case;
  end if;
end if;
end process;



slaves_generate:for I in 1 to 3 generate

slave_inst: entity work.ds1821_if_slave
port map (
      rst           => rst, -- in std_logic;
      clk           => clk, -- in std_logic;
      detect_sensor => detect_sensor, -- in std_logic;
      get_bit       => shift_bit_in, -- in std_logic;
      output_reading=> op_reading, -- in std_logic;
      temp          => temp(I), -- out std_logic_vector(7 downto 0);
      dq            => din(I) -- in std_logic
      );

end generate;




end rtl;
