-------------------------------------------------------------------------------
-- ad7918_if.vhd
--
-- Author : Nial Stewart, Nial Stewart Developments Ltd.
--          www.nialstewartdevelopments.co.uk
-- Date   : 21/03/2008
--
--
-------------------------------------------------------------------------------
-- This is a simple module to drive the AD7918 interface to the NSD General
-- Purpose Interface Board when attached to a CycloneIII starter board.
--
-- The input clock is assumed to be 80MHz (the master clock frequency in
-- the example FPGA build)
--
--
-------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity ad7918_if is
port (
      clk         : in  std_logic;
      rst         : in  std_logic;

      -- Config bits
      ip_range    : in std_logic; -- '0' = 0->2* Vref (ie 5V), '1' = 0->1.25V
      seq_end     : in std_logic_vector(2 downto 0);

      -- ADC If
      adc_sclk    : out std_logic;
      adc_cs_n    : out std_logic;
      adc_din     : out  std_logic;
      adc_dout    : in std_logic;

      --adc conversion results
      ch_0        : out std_logic_vector(9 downto 0);
      ch_1        : out std_logic_vector(9 downto 0);
      ch_2        : out std_logic_vector(9 downto 0);
      ch_3        : out std_logic_vector(9 downto 0);
      ch_4        : out std_logic_vector(9 downto 0);
      ch_5        : out std_logic_vector(9 downto 0);
      ch_6        : out std_logic_vector(9 downto 0);
      ch_7        : out std_logic_vector(9 downto 0);

      -- The following tick is driven once evey 8 samples to indicate
      -- the 8 values have been updates.
      ops_valid   : out std_logic

     );
end ad7918_if;

architecture rtl of ad7918_if is

constant coding               : std_logic := '1'; -- '0' = Twos compliment, '1' = Straight binary


-- Master process signal to drive the initial dummy conversions then driving
-- the conversions.
type master_state_typ is (IDLE, INIT1, INIT2, CONVERTING, VALUE_RXD);
signal master_state   : master_state_typ;

signal dout                   : std_logic_vector(15 downto 0);
signal do_transfer            : std_logic;
signal start_delay            : unsigned(15 downto 0);

-- Slave process to drive the AD7918 inteface signals.
type slave_state_typ is (IDLE, CS, TRANSFERRING, END_DELAY);

signal slave_state            : slave_state_typ;
signal tx_shift_reg           : std_logic_vector(15 downto 0);
signal rx_shift_reg           : std_logic_vector(15 downto 0);
signal bit_count              : integer range 0 to 31;
signal sclk_int               : std_logic;
signal cs_n                   : std_logic;
signal rx_tx_fin              : std_logic;


-- Tick process to generate the timing ticks for the slave process.
-- Master clock in = 80MHz, the sclk max is 20MHz, so divide by
-- 4 to get a tick rate of 20 MHz, the serial clock is then generated
-- at half that speed.

signal tick_count             : integer range 0 to 31;
signal tick                   : std_logic;


begin


-------------------------------------------------------------------------------
-- Master process
-------------------------------------------------------------------------------

process(clk,rst)
begin
if(rst = '1') then
  master_state <= IDLE;
  do_transfer          <= '0';
  dout                 <= (others => '0');
  start_delay          <= (others => '1');
elsif(rising_edge(clk)) then
  case master_state is
    when IDLE =>
      if(start_delay = x"0000") then
        dout          <= (others => '1');  -- Need two transfers of '1's out of reset
        do_transfer   <= '1';
        master_state  <= INIT1;
      else
        start_delay <= start_delay - 1;
      end if;

    when INIT1 =>
      if(rx_tx_fin = '1') then
        do_transfer   <= '1';
        master_state  <= INIT2;
      else
        do_transfer   <= '0';
      end if;

    when INIT2 =>
      if(rx_tx_fin = '1') then
        dout <= "110" & seq_end & "1110" & ip_range & coding & "0000";
        do_transfer   <= '1';
        master_state  <= CONVERTING;
      else
        do_transfer <= '0';
      end if;

    when CONVERTING =>
      if(rx_tx_fin = '1') then
        master_state <= VALUE_RXD;
      end if;
      do_transfer <= '0';
      ops_valid   <= '0';

    when OTHERS => -- VALUE_RXD
      do_transfer <= '1';
      master_state <= CONVERTING;
      dout <= "010" & seq_end & "1110" & ip_range & coding & "0000";     -- Only write control sequence firt time!

      case rx_shift_reg(15 downto 13) is
        when "000" => ch_0 <= rx_shift_reg(12 downto 3);
        when "001" => ch_1 <= rx_shift_reg(12 downto 3);
        when "010" => ch_2 <= rx_shift_reg(12 downto 3);
        when "011" => ch_3 <= rx_shift_reg(12 downto 3);
        when "100" => ch_4 <= rx_shift_reg(12 downto 3);
        when "101" => ch_5 <= rx_shift_reg(12 downto 3);
        when "110" => ch_6 <= rx_shift_reg(12 downto 3);
        when OTHERS => ch_7 <= rx_shift_reg(12 downto 3); -- "111"
                    ops_valid <= '1';

      end case;
  end case;
end if;
end process;

-------------------------------------------------------------------------------
-- Slave process
-------------------------------------------------------------------------------

process(clk,rst)
begin
if(rst = '1') then
  slave_state  <= IDLE;
  cs_n         <= '1';
  tx_shift_reg <= (others => '0');
  rx_shift_reg <= (others => '0');
  bit_count    <= 0;
  sclk_int     <= '1';
  rx_tx_fin    <= '0';
  tick_count   <= 30;
  tick         <= '0';
elsif(rising_edge(clk)) then
  case slave_state is
    when IDLE =>
      if(do_transfer = '1') then
        tx_shift_reg <= dout;
        bit_count    <= 0;
        cs_n         <= '0';
        slave_state  <= CS;
      end if;
      rx_tx_fin <= '0';

    when CS =>
      if(tick = '1') then
        bit_count <= 1;
        sclk_int  <= '0';
        slave_state <= TRANSFERRING;
      end if;
    when TRANSFERRING =>
      if(tick = '1') then
        if(sclk_int = '1') then
          if(bit_count = 16) then
            bit_count   <= 0;
            slave_state <= END_DELAY;
            cs_n        <= '1';
          else
            bit_count <= bit_count + 1;
            sclk_int  <= '0';
          end if;
        else
          tx_shift_reg(15 downto 1) <= tx_shift_reg(14 downto 0);
          rx_shift_reg(15 downto 1) <= rx_shift_reg(14 downto 0);
          rx_shift_reg(0) <= adc_dout;
          sclk_int        <= '1';
        end if;
      end if;

    when OTHERS => --END_DELAY
      if(tick = '1') then
        if(bit_count = 7) then
          rx_tx_fin   <= '1';
          slave_state <= IDLE;
        else
          bit_count <= bit_count + 1;
        end if;
      end if;
  end case;

  -- Ganerate the tick to drive this process
  if(slave_state = IDLE) then
    --tick_count <= 30;
    tick_count <= 2;
    tick       <= '0';
  elsif(tick_count = 0) then
    --tick_count <= 31;
    tick_count <= 3;
    tick       <= '1';
  else
    tick_count <= tick_count - 1;
    tick       <= '0';
  end if;

end if;
end process;

adc_sclk  <= sclk_int;
adc_cs_n  <= cs_n;
adc_din   <= tx_shift_reg(15);


end rtl;



