--------------------------------------------------------------------------
-- ad5318_if.vhd
--
-- Author : Nial Stewart, Nial Stewart Developments Ltd
--          www.nialstewartdevelopments.co.uk
-- Date   : 31/07/2007
--
--
--------------------------------------------------------------------------
-- A very simple module to implement the SPI interface to the AD5308 8bit
-- 8 channel ADC.
--
-- The 16 bit data word is sent to the ADC, bit allocations as follows...
-- (15)     - '0' = Data write, '1' = Control word write
-- (14..12) - Ch address
-- (11..2)  - 10 bit output value.
--
-- To write Half scale to ch 0 = 0x0800
-- To write Full scale to ch 3 = 0x3FFC
--
--------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity ad5318_if is
  port(
      clk         : in std_logic;
      rst         : in std_logic;

      dac_wr_data : in std_logic_vector(15 downto 0);
      dac_wr      : in std_logic;

      sclk        : out std_logic;
      sdout       : out std_logic;
      synch       : out std_logic;
      ldac_n      : out std_logic;
      busy        : out std_logic
      );
end ad5318_if;


architecture rtl of ad5318_if is

type  if_state_type is (IDLE,TXING,LOAD,END_DELAY1,END_DELAY2);
signal if_state   : if_state_type;


signal tick         : std_logic;
signal tick_count   : integer range 0 to 31;

signal tx_shift_reg : std_logic_vector(15 downto 0);

signal bit_count    : integer range 0 to 15;
signal sclk_int     : std_logic;

begin


-------------------------------------------------------------------------------
-- Tx Process;
-------------------------------------------------------------------------------

sdout <= tx_shift_reg(15);
sclk  <= sclk_int;

busy <= '0' when if_state = IDLE else '1';


process(clk,rst)
begin
if(rst = '1') then
  if_state      <= IDLE;
  sclk_int      <= '1';
  synch         <= '1';
  tx_shift_reg  <= (others => '0');
  ldac_n        <= '1';
elsif(rising_edge(clk)) then
  case if_state is
    when IDLE =>
      if(dac_wr = '1') then
        tx_shift_reg <= dac_wr_data;
        synch        <= '0';
        bit_count    <= 15;
        if_state     <= TXING;
      end if;

    when TXING =>

      if(tick = '1') then
        if(sclk_int = '0') then
          if(bit_count = 0) then
            synch  <= '1';
            if_state <= LOAD;
            ldac_n   <= '0';
            sclk_int <= '1';
          else
            tx_shift_reg(15 downto 1) <= tx_shift_reg(14 downto 0);
            bit_count <= bit_count - 1;
            sclk_int   <= '1';
          end if;
        else
          sclk_int <= '0';
        end if;
      end if;

    when LOAD =>
      if(tick = '1') then
        ldac_n   <= '1';
        if_state <= END_DELAY1;
      end if;

    when END_DELAY1 => --
      if(tick = '1') then
        if_state <= END_DELAY2;
      end if;

    when OTHERS => -- END_DELAY2
      if(tick = '1') then
        if_state <= IDLE;
      end if;
  end case;

end if;
end process;




-- Tick gen process - gives a tick at 20MHz.
process(clk,rst)
begin
if(rst = '1') then
  tick   <= '0';
  tick_count <= 7;
elsif(rising_edge(clk)) then
  if(if_state = IDLE) then
    tick_count <= 6; -- Will take one clock count to detect we're out of IDLE.
    tick       <= '0';
  elsif(tick_count = 0) then
    tick_count <= 7;
    tick       <= '1';
  else
    tick_count <= tick_count - 1;
    tick       <= '0';
  end if;

end if;
end process;

end rtl;




